﻿'---------------------------------------------------------------------------------------
' Module    : Data.vb
' Author    : A. De Filippo - The Labeljoy Team
' Date      : 2012-09-24
' Purpose   : Manage data
' License   : Free for private use. Do not reproduce any of this code without written permission.
'             Contact us first at: support@labeljoy.com
' Disclaimer: Use at your own risk.
'---------------------------------------------------------------------------------------

Imports System.Data.Odbc
Imports System.IO

Module Data

    ''' <summary>
    ''' Assemble data path and MDB file name
    ''' </summary>
    ''' <value></value>
    ''' <returns>The expected MDB file location</returns>
    Public ReadOnly Property MdbFileLocation() As String
        Get
            Return My.Computer.FileSystem.CombinePath(DataPath, "Logistic label.mdb")
        End Get
    End Property

    ''' <summary>
    ''' Assemble data path and LPA file name
    ''' </summary>
    ''' <value></value>
    ''' <returns>The default LPA file location</returns>
    Public ReadOnly Property LpaFileLocation() As String
        Get
            Return My.Computer.FileSystem.CombinePath(DataPath, "Logistic label.lpa")
        End Get
    End Property

    ''' <summary>
    ''' Build the data path inside current user's Document/Labeljoy folder.
    ''' </summary>
    ''' <value></value>
    ''' <returns>The full data path.</returns>
    Public ReadOnly Property DataPath() As String
        Get

            Dim sPath As String = String.Empty
            Try

                sPath = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments)
                sPath = Path.Combine(sPath, "Labeljoy")
                If Directory.Exists(sPath) = False Then Return String.Empty
            Catch ex As Exception
                ' Error -> Do nothing
            End Try

            ' OK
            Return sPath

        End Get

    End Property

    ''' <summary>
    ''' Load data from the MDB file
    ''' </summary>
    ''' <param name="Query">The SELECT SQL statement.</param>
    ''' <returns>A dataset containing the data.</returns>
    Public Function SqlSelect(ByVal Query As String) As DataSet

        ' Working objects
        Dim oCon As OdbcConnection = Nothing
        Dim objSqlEx As New DataSet
        Dim oAdr As OdbcDataAdapter = Nothing
        Dim oCmd As OdbcCommand = Nothing

        Try

            ' Open connection
            oCon = New OdbcConnection("Driver={Microsoft Access Driver (*.mdb)};Dbq=" & MdbFileLocation & ";Uid=Admin;Pwd=;")
            oCon.Open()

            ' Get data
            oCmd = New OdbcCommand(Query, oCon)
            oAdr = New OdbcDataAdapter(oCmd)
            oAdr.Fill(objSqlEx)
            oAdr.Dispose()
            Return objSqlEx

        Catch ex As Exception

            ' Error
            Throw New Exception(ex.Message, ex.InnerException)

        Finally

            ' Clean up
            If oCmd IsNot Nothing Then
                oCmd.Dispose()
                oCmd = Nothing
            End If
            If oAdr IsNot Nothing Then
                oAdr.Dispose()
                oAdr = Nothing
            End If
            If oCon IsNot Nothing Then
                oCon.Close()
                oCon.Dispose()
                oCon = Nothing
            End If
            GC.Collect()

        End Try

    End Function

    ''' <summary>
    ''' Update data in the MDB file.
    ''' </summary>
    ''' <param name="Query">The UPDATE SQL statement.</param>
    Public Sub SqlUpdate(ByVal Query As String)

        ' Working connection
        Dim oCon As OdbcConnection = Nothing
        Dim oCmd As OdbcCommand = Nothing

        Try

            ' Open connection
            oCon = New OdbcConnection("Driver={Microsoft Access Driver (*.mdb)};Dbq=" & MdbFileLocation & ";Uid=Admin;Pwd=;")
            oCon.Open()

            ' Init command
            oCmd = New OdbcCommand(Query, oCon)

            ' Execute
            oCmd.ExecuteNonQuery()

        Catch ex As Exception

            ' Error
            Throw New Exception(ex.Message, ex.InnerException)

        Finally

            ' Clean up
            If oCmd IsNot Nothing Then
                oCmd.Dispose()
                oCmd = Nothing
            End If
            If oCon IsNot Nothing Then
                oCon.Close()
                oCon.Dispose()
                oCon = Nothing
            End If
            GC.Collect()

        End Try

    End Sub

    ''' <summary>
    ''' Text escaping.
    ''' </summary>
    ''' <param name="Text">The string to be escaped.</param>
    ''' <returns>A copy of the input string with escaping where required.</returns>
    Public Function DbText(ByVal Text As String) As String
        If Len(Text) Then
            Text = Text.Replace("'", "''")
            Text = Text.Replace("\", "\\")
        End If
        Return "'" & Text & "'"
    End Function

    ''' <summary>
    ''' Properly clean up a dataset.
    ''' </summary>
    ''' <param name="oDts">The dataset to be cleaned up</param>
    Public Sub DtsClean(ByVal oDts As DataSet)
        If oDts IsNot Nothing Then
            oDts.Clear()
            oDts.Dispose()
            oDts = Nothing
        End If
    End Sub

    ''' <summary>
    ''' Set up data source for a ListControl (ComboBox, ListBox)
    ''' </summary>
    ''' <param name="Query">The SELECT SQL statement.</param>
    ''' <param name="lst">The ListControl (ComboBox or ListBox).</param>
    ''' <param name="DisplayMember">The field to display in the bound control.</param>
    ''' <param name="SelectedItem">The currently selected item in the list.</param>
    ''' <param name="ValueMember">The ValueMember, if any. Normally the primary index.</param>
    ''' <returns>The number of rows found.</returns>
    Public Function SetListDataSource(ByVal Query As String,
                                      ByRef lst As ListControl,
                                      ByVal DisplayMember As String,
                                      Optional ByVal SelectedItem As String = "",
                                      Optional ByVal ValueMember As String = "") As UInteger
        CleanListDataSource(lst, True)
        Dim oDts As DataSet = SqlSelect(Query)
        If oDts Is Nothing Then Exit Function
        If oDts.Tables.Count = 0 Then Exit Function
        With lst
            .DataSource = oDts.Tables(0)
            .DisplayMember = DisplayMember
            .Text = SelectedItem
            .ValueMember = ValueMember
        End With
        ' Result
        Return oDts.Tables(0).Rows.Count
    End Function

    ''' <summary>
    ''' Properly clean up binding source
    ''' </summary>
    ''' <param name="lst">The ListControl (ComboBox or ListBox).</param>
    ''' <param name="bSkipListDispose">Flag to avoid disposing the control after clean up. Set to True if you're re-loading data.</param>
    Public Sub CleanListDataSource(ByRef lst As ListControl,
                                   Optional ByVal bSkipListDispose As Boolean = False)
        Try
            Dim oDtt As DataTable = lst.DataSource
            If oDtt Is Nothing = False Then
                oDtt.Clear()
                oDtt.Dispose()
                oDtt = Nothing
            End If
            If bSkipListDispose = False Then lst.Dispose()
            GC.Collect()
        Catch ex As Exception
            ' Do nothing
        End Try
    End Sub

End Module